# pylint: disable=unused-import,abstract-method,unused-argument,no-member

##########################################################################################
#### Imports
##########################################################################################

import logging
import rpdb
import random
import re
import time
import os
import warnings
import json

from math import ceil
from typing import Union, List
from dallinger import db
from flask import Markup
from statistics import mean
from sqlalchemy import exc as sa_exc

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__file__)

import psynet.experiment
from psynet.timeline import (
    Timeline,
    join,
    PageMaker,
    MediaSpec
)
from psynet.page import (
    InfoPage,
    SuccessfulEndPage,
    NumberInputPage,
    NAFCPage,
    TextInputPage
)
from psynet.trial.non_adaptive import (
    StimulusSpec,
    StimulusVersionSpec,
    StimulusSet,
    NonAdaptiveTrialMaker,
    NonAdaptiveTrial
)
from psynet.modular_page import ModularPage, Prompt, NAFCControl, VideoSliderControl
from psynet.ishihara import colour_blind_test
from psynet.colour_vocab import colour_vocab_test

CONFIG = {
    "num_experiment_trials": 80,
    "num_repeat_trials": 8
}

class VideoFramePrompt(Prompt):
    macro = "video_frame"
    external_template = "video_frame_prompt.html"

    def __init__(
            self,
            *,
            url: str,
            file_type: str,
            width: str,
            height: str,
            starting_value: float = 0.5,
            minimal_time: float = 2.0,
            **kwargs
        ):
        super().__init__(**kwargs)

        assert 0 <= starting_value and starting_value <= 1

        self.url = url
        self.file_type = file_type
        self.width = width
        self.height = height
        self.starting_value = starting_value
        self.minimal_time = minimal_time

    @property
    def metadata(self):
        return {
            **super().metadata,
            "url": self.url,
            "starting_value": self.starting_value,
            "minimal_time": self.minimal_time
        }

    @property
    def media(self):
        return MediaSpec(video={"slider_video": self.url})

demographics = join(
    InfoPage(
        "First we need to ask some quick questions about you.",
        time_estimate=5
    ),
    NumberInputPage(
        label='age',
        prompt='What is your age, in years?',
        time_estimate=5
    ),
    NAFCPage(
        label='gender',
        prompt='With what gender do you most identify yourself?',
        time_estimate=5,
        choices=['Male', 'Female', 'Other'],
        arrange_vertically=True
    ),
    NAFCPage(
        label='education',
        prompt='What is your highest educational qualification?',
        time_estimate=7,
        choices=['None', 'Elementary school', 'Middle school', 'High school', 'Bachelor', 'Master', 'PhD'],
        arrange_vertically=True
    ),
    TextInputPage(
        "country",
        "What country are you from?",
        time_estimate=5,
        one_line=True
    ),
    TextInputPage(
        "mother_tongue",
        "What is your mother tongue?",
        time_estimate=5,
        one_line=True
    )
)

final_questionnaire = join(
    TextInputPage(
        "strategy",
        """
        Please tell us in a few words about your experience taking the task.
        What was your strategy?
        Did you find the task easy or difficult?
        Did you find it interesting or boring?
        """,
        time_estimate=20,
        one_line=False
    ),
    TextInputPage(
        "technical",
        """
        Did you experience any technical problems during the task?
        If so, please describe them.
        """,
        time_estimate=10,
        one_line=False
    )
)

def make_timeline():
    return Timeline(
        demographics,
        colour_vocab_test(),
        instructions(),
        InfoPage(
            f"""
            You will take up to
            {CONFIG['num_experiment_trials'] + CONFIG['num_repeat_trials']} trials
            where you have to answer this question. Remember to pay careful attention
            in order to get the best bonus!
            """,
            time_estimate=5
        ),
        make_experiment_trials(),
        final_questionnaire,
        SuccessfulEndPage()
    )

def instructions():
    return join(
        InfoPage(Markup(
            """
            <p>
                In each trial of this experiment you will be presented with a word
                and a face. Your task will be to judge how well the face
                matches the word.
            </p>
            <p>
                You will have four response options, ranging from 'Not at all' to 'Very much'.
                Choose the one you think is most appropriate.
            </p>
            """),
            time_estimate=5
        ),
        InfoPage(
            """
            The quality of your responses will be automatically monitored,
            and you will receive a bonus at the end of the experiment
            in proportion to your quality score. The best way to achieve
            a high score is to concentrate and give each trial your best attempt.
            """,
            time_estimate=5
        )
    )

def get_stimulus_set(phase: str):
    import pandas as pd
    phase = "experiment"
    df = pd.read_csv("states-for-validation.csv")

    targets = [
        "attractive",
        "fun",
        "intelligent",
        "serious",
        "trustworthy",
        "youthful"
    ]

    stimuli = [
        StimulusSpec(
            {
                "old_network_id": record["network_id"],
                "old_node_id": record["node_id"],
                "degree": record["degree"],
                "iteration": record["iteration"],
                "original_target": record["target"],
                "new_target": record["target"], # here you could substitute different targets if you wanted
                "video_url": record["video_url"],
                "vector": json.loads(record["vector"]),
                "active_index": record["active_index"],
                "active_value": record["active_value"],
                "slider_position": record["slider_position"]
            },
            phase=phase
        )
        for record in df.to_dict("records")
        # for target in targets
    ]
    return StimulusSet(stimuli)

nafc_choices = [
    (1, "(1) Not at all"),
    (2, "(2) A little"),
    (3, "(3) Quite a lot"),
    (4, "(4) Very much")
]

class RatingControl(NAFCControl):
    def __init__(self):
        super().__init__(
            choices=[x[0] for x in nafc_choices],
            labels=[x[1] for x in nafc_choices],
            min_width="125px"
        )

    def format_answer(self, raw_answer, **kwargs):
        return int(raw_answer)

class CustomTrial(NonAdaptiveTrial):
    __mapper_args__ = {"polymorphic_identity": "custom_trial"}

    def show_trial(self, experiment, participant):
        prompt = Markup(
            f"""
            <p>How well does the face match the following word:</p>
            <p><strong>{self.definition['new_target']}</strong></p>
            """
        )

        return ModularPage(
            "custom_trial",
            VideoFramePrompt(
                text=prompt,
                text_align="center",
                url=self.definition["video_url"],
                file_type="mp4",
                width="350px",
                height="350px",
                minimal_time=1,
                starting_value=self.definition["slider_position"]
            ),
            RatingControl()
        )

class CustomTrialMaker(NonAdaptiveTrialMaker):
    give_end_feedback_passed = False
    performance_threshold = -1.0

    def compute_bonus(self, score, passed):
        if self.phase == "practice":
            return 0.0
        elif self.phase == "experiment":
            if score is None:
                return 0.0
            else:
                return max(0.0, score)
        else:
            raise NotImplementedError

def make_experiment_trials():
    return CustomTrialMaker(
        trial_class=CustomTrial,
        phase="experiment",
        stimulus_set=get_stimulus_set(phase="experiment"),
        time_estimate_per_trial=5,
        new_participant_group=True,
        recruit_mode="num_trials",
        target_num_participants=None,
        target_num_trials_per_stimulus=50,
        max_trials_per_block=CONFIG["num_experiment_trials"],
        allow_repeated_stimuli=False,
        max_unique_stimuli_per_block=None,
        active_balancing_within_participants=True,
        active_balancing_across_participants=True,
        check_performance_at_end=True,
        check_performance_every_trial=False,
        fail_trials_on_premature_exit=True,
        fail_trials_on_participant_performance_check=False,
        num_repeat_trials=CONFIG["num_repeat_trials"]
    )

class Exp(psynet.experiment.Experiment):
    timeline = make_timeline()

    def __init__(self, session=None):
        super().__init__(session)
        self.initial_recruitment_size = 20

extra_routes = Exp().extra_routes()
